% Matlab code for the simulation of the minimal physical model presented in "Persistent cell migration emerges from a coupling between protrusion dynamics and polarized trafficking by Vaidiulyte et al.
% The code can be run to reproduce the lookup tables presented in the
% manuscript
% Written by M. Coppey, 03-25-2021
%% parameters of the model
clear all
tt = 1000; % total duration of the simulation
L = 101; % number of markers along the cell contour
pp = 0.25; % probability of protrusion per unit of time
simgaNG = 0.8; % spatial extent of the protrusion bias toward the polarity axis
speedNG = 0.5; % speed of the polarity axis reorientation
feed1 = 1; % feedback polarity axis --> protrusion on =1, off =0
feed2 = 1; % feedback protrusion --> polarity axis on =1, off =3
pGi=1; % probability to make a protrusion in front of the polarity axis
NGi=1; % bias of the polarity axis towards protrusions

%% simulations
%------------------------------------ loop over the parameters pG and NG
for pG = 0:0.05:1; % probability to make a protrusion in front of the polarity axis
    NGi=1;
    for NGforce = 0:0.05:1; % bias of the polarity axis towards protrusions
        Nrep = 20; % number of repetition
        
        % Step 1: synthetic morphodynamic map for a single protrusion
        % Transfer matrix
        Ttot = 100;
        % create transfer matrix
        x = -50:50;
        tt = 0:Ttot;
        H = zeros(Ttot+1,101); % H transfer function
        c = 0.5; % speed
        f1 = 0.1; % fade factor 1
        f2 = 0.1; % fade factor 2
        f3 = 0.2; % fade factor 3
        for t=0:Ttot
            % initial lateral inhibition:
            gauss1 = -0.01*exp(-f1*t);
            % traveling positive waves:
            gauss2 = (0.5.*normpdf(x,c.*t,5)+0.5.*normpdf(x,-c.*t,5))*exp(-f2*t);
            % central long lasting inhibition
            gauss3 = -(0.1.*normpdf(x,0,3).*normpdf(t,35,25).*sqrt(2*pi).*20);
            % central very peaked positve function
            gauss4 = 0.1.*normpdf(x,0,1).*normpdf(t,10,5).*sqrt(2*pi);
            h0 = gauss2+gauss1+gauss3+gauss4;
            H(t+1,:)=h0;
        end
        % ensure zero mean
        int = sum(sum(H));
        H = H-int./numel(H);
        maxt=max(max(H));
        Hf = flipud(H);
        % convolve with signal
        signal= zeros(Ttot+1,101); % H transfer function
        f1 = 0.1; % time scale to appear
        f2 = 1; % duration
        f3 = 3; % fading time scale
        for t=0:Ttot
            if t<f2
                signal(t+1,:)= (1-exp(-f1.*t)).*normpdf(x,0,5);
            else
                signal(t+1,:)= exp(-f3.*(t-f2)).* (1-exp(-f1.*t)).*normpdf(x,0,5);
            end
        end
        % ensure 1 mean
        int = sum(sum(signal));
        signal = signal-int./numel(signal);
        M = conv2(signal,H);
        M = M(1:Ttot+1,50:150);
        H = M;
        int = sum(sum(H));
        H = H-int./numel(H);
        
        %---------------------- loop over the number of realizations
        for nr=1:Nrep
            % Step 2: simulate the whole morphodynamic map 
            tt = 1000; % total duration of the simulation
            L = 101; % number of markers along the contour
            Mk = zeros(L,tt);
            pos = [];
            np = 0; % number of protrusions
            NG = zeros(1,tt);
            NG(1) = ceil(L*rand); % position of the NG axis, random
            NGprev = NG(1);
            % generate NG axis natural evolution of positioning
            va = smooth(10*L*rand(tt+1,1),100,'rloess');
            NGn = diff(va);
            %------------------------------ main loop over time
            for t=1:tt 
                NG(t) = NGprev;
                if rand<pp % compute if a new protrusion is made
                    if feed1==1
                        if rand>pG
                            posloc = ceil(L*rand); % random
                        else
                            posloc = round(normrnd(NG(t),simgaNG)); % in front of the polarity axis
                        end
                        if posloc>L
                            posloc=posloc-L;
                        elseif posloc<1
                            posloc = L+posloc;
                        end
                        pos = [pos;posloc,t]; % feedback on
                        np =np+1;
                    end
                end
                % compute the map given the protrusions
                for k=1:np
                    pro = pos(k,:); % current protrusion
                    if t-pro(2)<Ttot+1 %time of the protrusion
                        locM = H(t-pro(2)+1,:);
                        locM = circshift(locM,50+pro(1));
                        Mk(:,t) = Mk(:,t)+locM';
                    end
                end
                % update polarity axis positioning
                if feed2==1 
                    dF = 0;
                    for k=1:L
                        if (abs(k-NG(t))>=L-abs(NG(t)-k))
                            distpt = L-abs(NG(t)-k);
                            if Mk(k,t)>=0
                                dF(k) = -sign(k-NG(t))*Mk(k,t);
                            end
                        else
                            distpt = abs(k-NG(t));
                            if Mk(k,t)>=0
                                dF(k) = sign(k-NG(t))*Mk(k,t); 
                            end
                        end
                    end
                    NG(t) = NG(t)+speedNG*(NGforce*sum(dF)+(1-NGforce)*NGn(t));
                    if NG(t)>L
                        NG(t)=NG(t)-L;
                    elseif NG(t)<1
                        NG(t) = L+NG(t)-1;
                    end
                    NGprev = NG(t);
                end
            end
            %------------------------------  end of main loop over time
           
            % Step 3: make a trajectory from the morphomap
            angles = [0:2*pi/100:2*pi];
            dx=cos(angles);
            dy=sin(angles);
            posx = 0;
            posy = 0;
            for i=1:tt
                dispx(i) = sum(dx'.*Mk(:,i));
                dispy(i) = sum(dy'.*Mk(:,i));
                norm = sqrt( dispx(i)^2+ dispy(i)^2);
                if norm~=0
                    dispx(i) =  dispx(i)./norm;
                    dispy(i) =  dispy(i)./norm;
                end
                posx(i+1) = posx(i)+sum(dx'.*Mk(:,i));
                posy(i+1) = posy(i)+sum(dy'.*Mk(:,i));
            end
            Px(:,nr)=posx;
            Py(:,nr)=posy;

            % Step 4: calculate the protrusion unicity index
            mks = sort(Mk(:));
            th =  0.0001;
            BW = im2bw(Mk,th);
            bins = 10;
            for i=1:1000-bins
                BWloc = BW(:,i:i+bins);
                CC = bwconncomp(BWloc);
                NPloc(i)=CC.NumObjects;
                if sum(BWloc(1,:))>=1&sum(BWloc(end,:))>=1
                    NPloc(i)=NPloc(i)-1;
                end
            end
            NP(nr)=mean(NPloc); % mean number of competing protrusions
            
            % Step 5: calculate the alignment index
            for i=1:tt
                dgx(i) = dx(ceil(NG(i)));
                dgy(i) = dy(ceil(NG(i)));
                va = [dgx(i)   dgy(i)];
                vb = [dispx(i)   dispy(i)];
                aloc(i) = atan2(va(1)*vb(2)-va(2)*vb(1),va(1)*vb(1)+va(2)*vb(2));
            end
            X = sum(cos(aloc))./tt;
            Y = sum(sin(aloc))./tt;
            r(nr) = sqrt(X^2+Y^2); % angular dispersion     
        end
        %------------------------------  end of  loop over realizations
        
        % Step 6: calculate the persistence time
        for nr=1:Nrep
            for n=1:tt/20
                vac(n,nr) = 0;
                count = 0;
                for i=1:tt-n
                    normloc = (((Px(i,nr)-Px(i+1,nr))^2+(Py(i,nr)-Py(i+1,nr))^2)*((Px(i+n,nr)-Px(i+n+1,nr))^2+(Py(i+n,nr)-Py(i+n+1,nr))^2));
                    if normloc~=0
                        vac(n,nr)=vac(n,nr)+((Px(i,nr)-Px(i+1,nr))*(Px(i+n,nr)-Px(i+n+1,nr))+(Py(i,nr)-Py(i+1,nr))*(Py(i+n,nr)-Py(i+n+1,nr)))./sqrt(normloc);
                        count = count+1;
                    end
                end
                vac(n,nr)=vac(n,nr)/count;
            end
        end
        % compute frame at which correlation is 0.5
        maxc = tt/20;
        xx = 1:tt/20;
        for nr=1:Nrep
            vm(nr)=interp1(vac(:,nr),xx,0.5);
            if isnan(vm(nr))
                vm(nr)=maxc;
            end
        end
        
        % Save results
        RES1(pGi,NGi)=mean(vm); % persistence time
        RES2(pGi,NGi)=mean(NP); % number of competing protrusions
        RES3(pGi,NGi)=mean(r); % angular dispersion
        NGi = NGi+1;
        NGforce
    end
    pG
    pGi = pGi+1;
end
%% plots
% persistence time
K = (1/25)*ones(5);
RES1S = conv2(RES1,K,'valid');
figure(1)
imshow(rot90(RES1S.*5./60),[],'initialmagnification',5000);colormap('jet');colorbar;set(gca,'FontSize',24,'fontweight','bold')
title('Persistence time (hrs)')
saveas(gcf,'persistence.png')

% protrusion unicity
% K = (1/25)*ones(5);
RES2S = conv2(1./RES2,K,'valid');
figure(2)
imshow(rot90(RES2S),[],'initialmagnification',5000);colormap('jet');colorbar;set(gca,'FontSize',24,'fontweight','bold')
title('Protrusive unicity (a.u.)')
saveas(gcf,'protrusions.png')

% alignement index
RES3S = conv2(RES3,K,'valid');
figure(3)
imshow(rot90(RES3S),[],'initialmagnification',5000);colormap('jet');colorbar;set(gca,'FontSize',24,'fontweight','bold')
title('Alignement index (a.u.)')
saveas(gcf,'alignment.png')